#define LLOG(x)	 // DLOG(x)

thread_local int sWorkerId;
static Atomic sWorkerCount;
static Atomic sJobCancel;
static Mutex  mListMutex;
static Index<int64> sCancelList;
template<class T> static thread_local One<T>* sData = NULL;

static void UpdateList()
{
	if(!GetJobCount()) {
		Mutex::Lock __(mListMutex);
		sCancelList.Clear();
	}
	else
	if(IsJobCancelled()) {
		Mutex::Lock __(mListMutex);
		auto i = sCancelList.Find(sWorkerId);
		if(i >= 0) {
			sCancelList.Unlink(i); // A bit of optimization.
			LLOG("Worker #" << sWorkerId << ": Job cancelled.");
		}
	}
}

static int GetNewWorkerId()
{
	static int64 generator;
	if(generator == INT_MAX)
		generator = 0;
	else
		++generator;
	return generator;
}

int GetWorkerId()
{
	return sWorkerId;
}

int GetJobCount()
{
	return sWorkerCount;
}

void WaitForJobs()
{
	while(GetJobCount())
		Sleep(1);
}

void CancelJobs()
{
	sJobCancel++;
	WaitForJobs();
	sJobCancel--;
}

bool IsJobCancelled()
{
	if(Thread::IsShutdownThreads() || sJobCancel)
		return true;
	Mutex::Lock __(mListMutex);
	return sCancelList.Find(sWorkerId) >= 0;
}

template<class T>
void Job<T>::Init()
{
	work = Null;
	status = IDLE;
	wid = GetNewWorkerId();
	error = MakeTuple<int, String>(0, Null);
	Clear();

	auto b = worker.Run([=]{
		sWorkerId  = wid;
		sData<T> = &data;
		Work();
		sData<T> = NULL;
	});
	if(!b)  {
		status = FAILED;
		throw JobError("Worker initializetion failed!");
	}
	LLOG("Worker #" << wid << ": Initialized.");

}

template<class T>
void Job<T>::Exit()
{
	while(!IsFinished());
	{
		lock.Enter();
		status = EXIT;
		lock.Leave();
		cv.Signal();
	}
	LLOG("Worker #" << wid << ". Exit signal sent...");
	worker.Wait();
}

template<class T>
void Job<T>::Work()
{
	Mutex::Lock __(lock);
	for(;;) {
		while(status != WORKING) { // Takes care of spurious wake-ups
			if(status == EXIT) {
				LLOG("Worker #" << wid << ": Exit signal received. Shutting down job.");
				return;
			}
			LLOG("Worker #" << wid << ": Waiting for work.");
			cv.Wait(lock);
			LLOG("Worker #" << wid << ": Waiting is ended.");
		}
		AtomicInc(sWorkerCount);
		try {
			LLOG("Worker #" << wid << ": Working...");
			work();
			status = FINISHED;
			LLOG("Worker #" << wid << ": Finished!");
		}
		catch(JobError& e) {
			status = FAILED;
			error = MakeTuple<int, String>(e.code, e);
			LLOG("Worker #" << wid << ": Failed. " << e);
		}
		AtomicDec(sWorkerCount);
		UpdateList();
	}
}

template<class T>
bool Job<T>::Start(Event<>&& fn)
{
	auto b  = IsFinished();
	if(b) {
		LLOG("Worker #" << wid << ": Starting to work.");
		work = pick(fn);
		lock.Enter();
		status = WORKING;
		lock.Leave();
		cv.Signal();
	}
	else LLOG("Worker #" << wid << ": Couldn't start working. Worker is busy.");
	return b;
}

template<class T>
void Job<T>::Cancel()
{
	Mutex::Lock __(mListMutex);
	sCancelList.FindAdd(wid);
}

template<class T>
Job<T>& Job<T>::Finish()
{
	while(!IsFinished())
		;	// Sleep(1);
	return *this;
}

template<class T>
bool Job<T>::IsFinished()
{
	Mutex::Lock __(lock);
	return status != WORKING;
}

template<class T>
T& Job<T>::Data()
{
	ASSERT(!Thread::IsMain());
	return **sData<T>;
}
